<?php
/**
 * Plugin Name:       Kokovoice
 * Plugin URI:        https://kokoro.lancesmith.cc/wpplugins/kokovoice
 * Description:       Embed the Kokoro voice generation form - Public access, works for all visitors
 * Version:           0.3.3
 * Author:            LanceSmith.cc
 * License:           GPLv2 or later
 * Text Domain:       kokovoice
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Kokovoice' ) ) {
	final class Kokovoice {
		const VERSION    = '0.3.3';
		const OPTION_KEY = 'kokovoice_settings';

		public static function init() {
			static $instance = null;
			if ( null === $instance ) {
				$instance = new self();
			}
		}

		private function __construct() {
			add_action( 'init', [ $this, 'register_assets' ] );
			add_action( 'init', [ $this, 'register_shortcode' ] );
			add_action( 'init', [ $this, 'register_block' ] );
			add_action( 'admin_init', [ $this, 'register_settings' ] );
			add_action( 'admin_menu', [ $this, 'register_settings_page' ] );
			add_action( 'rest_api_init', [ $this, 'register_rest_routes' ] );
		}

		public static function get_settings() {
			$defaults = [
				'api_key' => '',
				'voice'   => 'af_alloy',
				'model'   => 'model_q8f16',
				'speed'   => '1.0',
			];

			return wp_parse_args( get_option( self::OPTION_KEY, [] ), $defaults );
		}

		public function register_assets() {
			$plugin_dir = plugin_dir_url( __FILE__ );

			wp_register_style(
				'kokovoice-frontend',
				$plugin_dir . 'assets/css/kokovoice.css',
				[],
				self::VERSION
			);

			// PUBLIC VERSION: Use kokovoice-public.js (no nonce)
			wp_register_script(
				'kokovoice-frontend',
				$plugin_dir . 'assets/js/kokovoice-public.js',
				[],
				self::VERSION,
				true
			);

			wp_register_script(
				'kokovoice-block',
				$plugin_dir . 'assets/js/block.js',
				[ 'wp-blocks', 'wp-element', 'wp-i18n', 'wp-components', 'wp-block-editor' ],
				self::VERSION,
				true
			);
		}

		public function register_settings() {
			register_setting( self::OPTION_KEY, self::OPTION_KEY, [ $this, 'sanitize_settings' ] );

			add_settings_section(
				'kokovoice_settings_section',
				__( 'Kokoro API Configuration', 'kokovoice' ),
				function () {
					echo '<p>' . esc_html__( 'Store your Kokoro API credentials and defaults.', 'kokovoice' ) . '</p>';
					echo '<p><strong>' . esc_html__( 'Note:', 'kokovoice' ) . '</strong> ' . esc_html__( 'API key is optional. This plugin allows PUBLIC ACCESS (no WordPress login required).', 'kokovoice' ) . '</p>';
				},
				'kokovoice_settings'
			);

			add_settings_field(
				'kokovoice_api_key',
				__( 'Kokoro API Key (Optional)', 'kokovoice' ),
				[ $this, 'render_text_field' ],
				'kokovoice_settings',
				'kokovoice_settings_section',
				[
					'label_for'   => 'kokovoice_api_key',
					'option_key'  => 'api_key',
					'placeholder' => 'kw_...'
				]
			);

			add_settings_field(
				'kokovoice_voice',
				__( 'Default voice formula', 'kokovoice' ),
				[ $this, 'render_text_field' ],
				'kokovoice_settings',
				'kokovoice_settings_section',
				[
					'label_for'   => 'kokovoice_voice',
					'option_key'  => 'voice',
					'placeholder' => 'af_alloy'
				]
			);

			add_settings_field(
				'kokovoice_model',
				__( 'Model', 'kokovoice' ),
				[ $this, 'render_text_field' ],
				'kokovoice_settings',
				'kokovoice_settings_section',
				[
					'label_for'   => 'kokovoice_model',
					'option_key'  => 'model',
					'placeholder' => 'model_q8f16'
				]
			);

			add_settings_field(
				'kokovoice_speed',
				__( 'Default speed', 'kokovoice' ),
				[ $this, 'render_text_field' ],
				'kokovoice_settings',
				'kokovoice_settings_section',
				[
					'label_for'   => 'kokovoice_speed',
					'option_key'  => 'speed',
					'placeholder' => '1.0'
				]
			);
		}

		public function sanitize_settings( array $input ) {
			$settings = self::get_settings();
			$settings['api_key'] = isset( $input['api_key'] ) ? sanitize_text_field( $input['api_key'] ) : '';
			$settings['voice']   = isset( $input['voice'] ) ? sanitize_text_field( $input['voice'] ) : 'af_alloy';
			$settings['model']   = isset( $input['model'] ) ? sanitize_text_field( $input['model'] ) : 'model_q8f16';
			$settings['speed']   = isset( $input['speed'] ) ? sanitize_text_field( $input['speed'] ) : '1.0';
			return $settings;
		}

		public function render_text_field( array $args ) {
			$settings = self::get_settings();
			$key      = $args['option_key'];
			$value    = isset( $settings[ $key ] ) ? $settings[ $key ] : '';
			printf(
				'<input type="text" id="%1$s" name="%2$s[%3$s]" value="%4$s" class="regular-text" placeholder="%5$s" />',
				esc_attr( $args['label_for'] ),
				esc_attr( self::OPTION_KEY ),
				esc_attr( $key ),
				esc_attr( $value ),
				esc_attr( $args['placeholder'] ?? '' )
			);
		}

		public function register_settings_page() {
			add_options_page(
				__( 'Kokovoice', 'kokovoice' ),
				__( 'Kokovoice', 'kokovoice' ),
				'manage_options',
				'kokovoice-settings',
				[ $this, 'render_settings_page' ]
			);
		}

		public function render_settings_page() {
			if ( ! current_user_can( 'manage_options' ) ) {
				return;
			}
			?>
			<div class="wrap">
				<h1><?php esc_html_e( 'Kokovoice Settings', 'kokovoice' ); ?></h1>
				<form method="post" action="options.php">
					<?php
					settings_fields( self::OPTION_KEY );
					do_settings_sections( 'kokovoice_settings' );
					submit_button();
					?>
				</form>
				<p>
					<?php
						echo wp_kses_post( sprintf(
							__( 'Use the shortcode %1$s or the "Kokoro Voice Form" block.', 'kokovoice' ),
							'<code>[kokoro_voice_form]</code>'
						) );
					?>
				</p>
				<p style="background: #fff3cd; padding: 10px; border-left: 4px solid #ffc107;">
					<strong><?php esc_html_e( '⚠️ Public Access Enabled:', 'kokovoice' ); ?></strong>
					<?php esc_html_e( 'This version allows anyone to use the voice form without logging in. Perfect for public websites!', 'kokovoice' ); ?>
				</p>
			</div>
			<?php
		}

		public function register_shortcode() {
			add_shortcode( 'kokoro_voice_form', [ $this, 'render_form_shortcode' ] );
		}

		public function render_form_shortcode( array $atts ) {
			$settings = self::get_settings();
			
			$atts = shortcode_atts(
				[
					'button_label' => __( 'Generate Voice', 'kokovoice' ),
					'voice'        => $settings['voice'],
					'speed'        => $settings['speed'],
					'debug'        => 'false',
				],
				$atts,
				'kokoro_voice_form'
			);

			return $this->render_form_markup( $atts );
		}

		protected function render_form_markup( array $args ) {
			$debug_mode = filter_var( $args['debug'] ?? 'false', FILTER_VALIDATE_BOOLEAN );
			$this->enqueue_frontend_assets( $debug_mode );

			$defaults = self::get_settings();
			$voice    = $args['voice'] ?: $defaults['voice'];
			$speed    = $args['speed'] ?: $defaults['speed'];
			$button   = $args['button_label'] ?? __( 'Generate Voice', 'kokovoice' );
			$voice_options = $this->get_voice_options();
			if ( ! isset( $voice_options[ $voice ] ) ) {
				$voice_options = [ $voice => $voice ] + $voice_options;
			}

			ob_start();
			?>
			<div
				class="kokovoice"
				data-kokovoice-form
				data-default-voice="<?php echo esc_attr( $voice ); ?>"
				data-default-speed="<?php echo esc_attr( $speed ); ?>"
			>
				<form class="kokovoice__form">
					<label class="kokovoice__field">
						<span class="kokovoice__label"><?php esc_html_e( 'Text to generate', 'kokovoice' ); ?></span>
						<textarea
							name="text"
							rows="6"
							maxlength="4500"
							required
							placeholder="<?php echo esc_attr__( 'Enter text...', 'kokovoice' ); ?>"
						></textarea>
					</label>

					<div class="kokovoice__field-group">
						<label class="kokovoice__field">
							<span class="kokovoice__label"><?php esc_html_e( 'Voice', 'kokovoice' ); ?></span>
							<select name="voice" required>
								<?php foreach ( $voice_options as $value => $label ) : ?>
									<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $value, $voice ); ?>>
										<?php echo esc_html( $label ); ?>
									</option>
								<?php endforeach; ?>
							</select>
						</label>
						<label class="kokovoice__field">
							<span class="kokovoice__label"><?php esc_html_e( 'Speed', 'kokovoice' ); ?></span>
							<input type="number" name="speed" min="0.1" max="2" step="0.05" value="<?php echo esc_attr( $speed ); ?>" />
						</label>
					</div>

					<button type="submit" class="kokovoice__submit"><?php echo esc_html( $button ); ?></button>
				</form>
				<div class="kokovoice__status" role="status" aria-live="polite"></div>
				<audio class="kokovoice__audio" controls hidden></audio>
				<a class="kokovoice__download" download hidden><?php esc_html_e( 'Download MP3', 'kokovoice' ); ?></a>
			</div>
			<?php
			return ob_get_clean();
		}

		protected function enqueue_frontend_assets( $debug_mode = false ) {
			$settings = self::get_settings();

			wp_enqueue_style( 'kokovoice-frontend' );
			wp_enqueue_script( 'kokovoice-frontend' );

			// PUBLIC VERSION: No nonce needed for public access
			wp_localize_script(
				'kokovoice-frontend',
				'kokovoiceSettings',
				[
					'restUrl'      => esc_url_raw( rest_url( 'kokovoice/v1/generate' ) ),
					// NO nonce - public access version
					'hasApiKey'    => ! empty( $settings['api_key'] ),
					'maxCharacters'=> 4500,
					'debugMode'    => $debug_mode, // Enable debug logging if debug="true" in shortcode
					'labels'       => [
						'ready'   => __( 'Ready', 'kokovoice' ),
						'loading' => __( 'Generating...', 'kokovoice' ),
						'success' => __( 'Ready!', 'kokovoice' ),
						'error'   => __( 'Error generating audio', 'kokovoice' ),
						'empty'   => __( 'Please enter text', 'kokovoice' ),
					],
				]
			);
		}

		protected function get_voice_options() {
			$voices = [
				// English (US) - Female
				'af_heart'     => __( 'Heart - US Female (Best Quality)', 'kokovoice' ),
				'af_bella'     => __( 'Bella - US Female', 'kokovoice' ),
				'af_nicole'    => __( 'Nicole - US Female', 'kokovoice' ),
				'af_sarah'     => __( 'Sarah - US Female', 'kokovoice' ),
				'af_aoede'     => __( 'Aoede - US Female', 'kokovoice' ),
				'af_kore'      => __( 'Kore - US Female', 'kokovoice' ),
				'af_alloy'     => __( 'Alloy - US Female', 'kokovoice' ),
				'af_nova'      => __( 'Nova - US Female', 'kokovoice' ),
				'af_sky'       => __( 'Sky - US Female', 'kokovoice' ),
				'af_jessica'   => __( 'Jessica - US Female', 'kokovoice' ),
				'af_river'     => __( 'River - US Female', 'kokovoice' ),
				
				// English (US) - Male
				'am_fenrir'    => __( 'Fenrir - US Male', 'kokovoice' ),
				'am_michael'   => __( 'Michael - US Male', 'kokovoice' ),
				'am_puck'      => __( 'Puck - US Male', 'kokovoice' ),
				'am_echo'      => __( 'Echo - US Male', 'kokovoice' ),
				'am_eric'      => __( 'Eric - US Male', 'kokovoice' ),
				'am_liam'      => __( 'Liam - US Male', 'kokovoice' ),
				'am_onyx'      => __( 'Onyx - US Male', 'kokovoice' ),
				'am_santa'     => __( 'Santa - US Male', 'kokovoice' ),
				'am_adam'      => __( 'Adam - US Male', 'kokovoice' ),
				
				// English (GB) - Female
				'bf_emma'      => __( 'Emma - British Female', 'kokovoice' ),
				'bf_isabella'  => __( 'Isabella - British Female', 'kokovoice' ),
				'bf_alice'     => __( 'Alice - British Female', 'kokovoice' ),
				'bf_lily'      => __( 'Lily - British Female', 'kokovoice' ),
				
				// English (GB) - Male
				'bm_george'    => __( 'George - British Male', 'kokovoice' ),
				'bm_fable'     => __( 'Fable - British Male', 'kokovoice' ),
				'bm_lewis'     => __( 'Lewis - British Male', 'kokovoice' ),
				'bm_daniel'    => __( 'Daniel - British Male', 'kokovoice' ),
				
				// Spanish
				'ef_dora'      => __( 'Dora - Spanish Female', 'kokovoice' ),
				'em_alex'      => __( 'Alex - Spanish Male', 'kokovoice' ),
				'em_santa'     => __( 'Santa - Spanish Male', 'kokovoice' ),
				
				// Japanese
				'jf_alpha'     => __( 'Alpha - Japanese Female', 'kokovoice' ),
				'jf_gongitsune'=> __( 'Gongitsune - Japanese Female', 'kokovoice' ),
				'jf_tebukuro'  => __( 'Tebukuro - Japanese Female', 'kokovoice' ),
				'jf_nezumi'    => __( 'Nezumi - Japanese Female', 'kokovoice' ),
				'jm_kumo'      => __( 'Kumo - Japanese Male', 'kokovoice' ),
				
				// Chinese (Mandarin)
				'zf_xiaobei'   => __( 'Xiaobei - Chinese Female', 'kokovoice' ),
				'zf_xiaoni'    => __( 'Xiaoni - Chinese Female', 'kokovoice' ),
				'zf_xiaoxiao'  => __( 'Xiaoxiao - Chinese Female', 'kokovoice' ),
				'zf_xiaoyi'    => __( 'Xiaoyi - Chinese Female', 'kokovoice' ),
				'zm_yunjian'   => __( 'Yunjian - Chinese Male', 'kokovoice' ),
				'zm_yunxi'     => __( 'Yunxi - Chinese Male', 'kokovoice' ),
				'zm_yunxia'    => __( 'Yunxia - Chinese Male', 'kokovoice' ),
				'zm_yunyang'   => __( 'Yunyang - Chinese Male', 'kokovoice' ),
				
				// Hindi
				'hf_alpha'     => __( 'Alpha - Hindi Female', 'kokovoice' ),
				'hf_beta'      => __( 'Beta - Hindi Female', 'kokovoice' ),
				'hm_omega'     => __( 'Omega - Hindi Male', 'kokovoice' ),
				'hm_psi'       => __( 'Psi - Hindi Male', 'kokovoice' ),
				
				// Italian
				'if_sara'      => __( 'Sara - Italian Female', 'kokovoice' ),
				'im_nicola'    => __( 'Nicola - Italian Male', 'kokovoice' ),
				
				// Portuguese
				'pf_dora'      => __( 'Dora - Portuguese Female', 'kokovoice' ),
				'pm_alex'      => __( 'Alex - Portuguese Male', 'kokovoice' ),
				'pm_santa'     => __( 'Santa - Portuguese Male', 'kokovoice' ),
			];

			return apply_filters( 'kokovoice_voice_options', $voices );
		}

		public function register_block() {
			register_block_type(
				__DIR__ . '/block',
				[
					'render_callback' => function( array $attributes = [] ) {
						$settings = self::get_settings();
						$button   = $attributes['buttonLabel'] ?? __( 'Generate Voice', 'kokovoice' );
						$voice    = $attributes['defaultVoice'] ?? $settings['voice'];
						$speed    = $attributes['defaultSpeed'] ?? $settings['speed'];
						return $this->render_form_markup(
							[
								'button_label' => $button,
								'voice'        => $voice,
								'speed'        => $speed,
							]
						);
					},
				]
			);
		}

		public function register_rest_routes() {
			register_rest_route(
				'kokovoice/v1',
				'/generate',
				[
					'callback'            => [ $this, 'rest_generate_audio' ],
					'permission_callback' => '__return_true', // PUBLIC ACCESS - No auth required!
					'args'                => [
						'text' => [
							'required' => true,
							'type'     => 'string',
						],
						'voice' => [
							'required' => false,
							'type'     => 'string',
						],
						'speed' => [
							'required' => false,
							'type'     => 'number',
						],
					],
					'methods'             => WP_REST_Server::CREATABLE,
				]
			);
		}

		public function rest_generate_audio( WP_REST_Request $request ) {
			$settings = self::get_settings();

			$text = trim( wp_strip_all_tags( wp_unslash( $request->get_param( 'text' ) ) ) );
			if ( '' === $text ) {
				return new WP_Error( 'kokovoice_empty_text', __( 'Text required', 'kokovoice' ), [ 'status' => 400 ] );
			}

			$max_chars = 4500;
			if ( strlen( $text ) > $max_chars ) {
				$text = substr( $text, 0, $max_chars );
			}

			$voice = sanitize_text_field( $request->get_param( 'voice' ) );
			if ( '' === $voice ) {
				$voice = $settings['voice'];
			}

			$speed = (float) $request->get_param( 'speed' );
			if ( ! is_numeric( $request->get_param( 'speed' ) ) ) {
				$speed = (float) $settings['speed'];
			}
			$speed = max( 0.1, min( 2.0, $speed ) );

			// Build headers
			$headers = [ 'Content-Type' => 'application/json' ];
			if ( ! empty( $settings['api_key'] ) ) {
				$headers['Authorization'] = 'Bearer ' . $settings['api_key'];
			}

			$response = wp_remote_post(
				'https://kokoro.lancesmith.cc/api/v1/audio/speech',
				[
					'timeout' => 300, // 5 minutes for long text generation
					'headers' => $headers,
					'body'    => wp_json_encode(
						[
							'model'           => $settings['model'],
							'voice'           => $voice,
							'input'           => $text,
							'speed'           => $speed,
							'response_format' => 'mp3',
						]
					),
				]
			);

			if ( is_wp_error( $response ) ) {
				return new WP_Error( 'kokovoice_api_error', $response->get_error_message(), [ 'status' => 500 ] );
			}

			$code = wp_remote_retrieve_response_code( $response );
			if ( $code < 200 || $code >= 300 ) {
				$body_error = wp_remote_retrieve_body( $response );
				$error_msg = __( 'API error', 'kokovoice' );
				
				$error_data = json_decode( $body_error, true );
				if ( isset( $error_data['message'] ) ) {
					$error_msg .= ': ' . $error_data['message'];
				}
				
				return new WP_Error( 'kokovoice_bad_status', $error_msg, [ 'status' => $code ] );
			}

			$body      = wp_remote_retrieve_body( $response );
			$mime_type = wp_remote_retrieve_header( $response, 'content-type' );
			if ( empty( $mime_type ) ) {
				$mime_type = 'audio/mpeg';
			}

			return rest_ensure_response(
				[
					'audio'     => base64_encode( $body ),
					'mime_type' => $mime_type,
				]
			);
		}
	}
}

Kokovoice::init();
