(function () {
  const settings = window.kokovoiceSettings || {};
  if (!settings.restUrl) {
    return;
  }

  const managed = new WeakSet();
  const debugMode = settings.debugMode || false;

  function debugLog(...args) {
    if (debugMode) {
      console.log('[Kokovoice Debug]', ...args);
    }
  }

  function clampSpeed(value) {
    const num = Number.parseFloat(value);
    if (Number.isNaN(num)) {
      return 1.0;
    }
    return Math.min(2, Math.max(0.1, num));
  }

  function arrayBufferFromBase64(base64) {
    const binary = atob(base64);
    const len = binary.length;
    const bytes = new Uint8Array(len);
    for (let i = 0; i < len; i += 1) {
      bytes[i] = binary.charCodeAt(i);
    }
    return bytes.buffer;
  }

  function revokeUrl(container) {
    if (container._kokovoiceUrl) {
      URL.revokeObjectURL(container._kokovoiceUrl);
      container._kokovoiceUrl = null;
    }
  }

  async function generateAudio(container, form, fields) {
    const { textField, voiceField, speedField, submitButton, statusEl, audioEl, downloadEl } = fields;

    const rawText = textField.value.trim();
    if (!rawText) {
      statusEl.textContent = settings.labels?.empty || 'Add some text first.';
      statusEl.classList.add('kokovoice__status--error');
      return;
    }

    if (rawText.length > settings.maxCharacters) {
      statusEl.textContent = `Text truncated to ${settings.maxCharacters} characters.`;
      textField.value = rawText.slice(0, settings.maxCharacters);
    }

    const payload = {
      text: textField.value.trim(),
      voice: (voiceField.value || container.dataset.defaultVoice || '').trim(),
      speed: clampSpeed(speedField.value || container.dataset.defaultSpeed || 1.0),
    };

    debugLog('Payload:', payload);
    debugLog('API URL:', settings.restUrl);

    submitButton.disabled = true;
    statusEl.textContent = settings.labels?.loading || 'Generating audio...';
    statusEl.classList.remove('kokovoice__status--error');

    try {
    // PUBLIC VERSION: No authentication headers sent at all
    debugLog('Sending request to API...');
    const response = await fetch(settings.restUrl, {
      method: 'POST',
      credentials: 'same-origin',
      headers: {
        'Content-Type': 'application/json',
        // NO nonce header - public access
      },
      body: JSON.stringify(payload),
      });

      debugLog('Response status:', response.status);

      if (!response.ok) {
        let message = `${settings.labels?.error || 'Unable to generate audio.'} (${response.status})`;
        try {
          const data = await response.json();
          debugLog('Error response data:', data);
          if (data && data.message) {
            message = data.message;
          }
        } catch (err) {
          debugLog('Failed to parse error response:', err);
        }
        throw new Error(message);
      }

      const data = await response.json();
      debugLog('Success response data received, audio length:', data.audio?.length || 0);
      
      const buffer = arrayBufferFromBase64(data.audio);
      const blob = new Blob([buffer], { type: data.mime_type || 'audio/mpeg' });
      debugLog('Audio blob created, size:', blob.size);

      revokeUrl(container);
      const url = URL.createObjectURL(blob);
      container._kokovoiceUrl = url;

      audioEl.src = url;
      audioEl.hidden = false;
      downloadEl.href = url;
      downloadEl.download = 'kokoro-voice.mp3';
      downloadEl.hidden = false;

      statusEl.textContent = settings.labels?.success || 'Audio ready';
      statusEl.classList.remove('kokovoice__status--error');
      debugLog('Audio ready for playback');

      try {
        await audioEl.play();
        debugLog('Audio playback started');
      } catch (err) {
        debugLog('Autoplay blocked:', err.message);
        // autoplay might be blocked; ignore
      }
    } catch (error) {
      console.error('Kokovoice Error:', error);
      debugLog('Full error details:', error);
      statusEl.textContent = error.message || settings.labels?.error || 'Unable to generate audio.';
      statusEl.classList.add('kokovoice__status--error');
    } finally {
      submitButton.disabled = false;
    }
  }

  function attach(container) {
    if (managed.has(container)) {
      return;
    }

    const form = container.querySelector('form');
    if (!form) {
      return;
    }

    const fields = {
      textField: form.querySelector('textarea[name="text"]'),
      voiceField: form.querySelector('[name="voice"]'),
      speedField: form.querySelector('input[name="speed"]'),
      submitButton: form.querySelector('button[type="submit"]'),
      statusEl: container.querySelector('.kokovoice__status'),
      audioEl: container.querySelector('.kokovoice__audio'),
      downloadEl: container.querySelector('.kokovoice__download'),
    };

    if (!fields.textField || !fields.voiceField || !fields.speedField || !fields.submitButton) {
      return;
    }

    container.dataset.defaultVoice = container.dataset.defaultVoice || fields.voiceField.value || '';
    container.dataset.defaultSpeed = container.dataset.defaultSpeed || fields.speedField.value || '1.0';

    if (fields.statusEl) {
      fields.statusEl.textContent = settings.labels?.ready || 'Ready';
      fields.statusEl.classList.remove('kokovoice__status--error');
    }

    form.addEventListener('submit', (event) => {
      event.preventDefault();
      generateAudio(container, form, fields);
    });

    managed.add(container);
  }

  function init() {
    document.querySelectorAll('[data-kokovoice-form]').forEach((el) => attach(el));
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  window.addEventListener('beforeunload', () => {
    document.querySelectorAll('[data-kokovoice-form]').forEach((container) => revokeUrl(container));
  });
})();

