(function () {
	const settings = window.kokoreaderSettings || {};
	
	if (!settings.restUrl) {
		return;
	}

	const container = document.getElementById('kokoreader-floating-button');
	if (!container) {
		return;
	}

	const button = container.querySelector('.kokoreader-button');
	const statusEl = container.querySelector('.kokoreader-status');
	let currentAudio = null;
	let isGenerating = false;

	function showStatus(message, isError = false) {
		if (!statusEl) return;
		
		statusEl.textContent = message;
		statusEl.classList.toggle('error', isError);
		statusEl.classList.add('visible');
		
		setTimeout(() => {
			statusEl.classList.remove('visible');
		}, 3000);
	}

	function getSelectedText() {
		const selection = window.getSelection();
		const text = selection.toString().trim();
		return text;
	}

	function updateButtonState() {
		const hasSelection = getSelectedText().length > 0;
		container.classList.toggle('no-selection', !hasSelection);
	}

	function arrayBufferFromBase64(base64) {
		const binary = atob(base64);
		const len = binary.length;
		const bytes = new Uint8Array(len);
		for (let i = 0; i < len; i++) {
			bytes[i] = binary.charCodeAt(i);
		}
		return bytes.buffer;
	}

	async function speakSelectedText() {
		const text = getSelectedText();
		
		if (!text) {
			showStatus('Please select some text first', true);
			return;
		}

		if (text.length > 5000) {
			showStatus('Selected text is too long (max 5000 characters)', true);
			return;
		}

		// Stop any currently playing audio
		if (currentAudio) {
			currentAudio.pause();
			currentAudio = null;
		}

		isGenerating = true;
		button.disabled = true;
		button.classList.add('loading');
		showStatus('Generating speech...');

		try {
			const response = await fetch(settings.restUrl, {
				method: 'POST',
				headers: {
					'Content-Type': 'application/json',
				},
				body: JSON.stringify({ text }),
			});

			if (!response.ok) {
				let errorMsg = `Error ${response.status}`;
				try {
					const data = await response.json();
					if (data.message) {
						errorMsg = data.message;
					}
				} catch (e) {
					// Ignore JSON parse errors
				}
				throw new Error(errorMsg);
			}

			const data = await response.json();
			
			if (!data.audio) {
				throw new Error('No audio data received');
			}

			// Convert base64 to audio
			const buffer = arrayBufferFromBase64(data.audio);
			const blob = new Blob([buffer], { type: data.mime_type || 'audio/mpeg' });
			const url = URL.createObjectURL(blob);

			// Create and play audio
			currentAudio = new Audio(url);
			
			currentAudio.addEventListener('ended', () => {
				URL.revokeObjectURL(url);
				showStatus('Finished speaking');
			});

			currentAudio.addEventListener('error', (e) => {
				URL.revokeObjectURL(url);
				showStatus('Failed to play audio', true);
			});

			await currentAudio.play();
			showStatus('Speaking...');

		} catch (error) {
			console.error('Kokoreader error:', error);
			showStatus(error.message || 'Failed to generate speech', true);
		} finally {
			isGenerating = false;
			button.disabled = false;
			button.classList.remove('loading');
		}
	}

	// Event listeners
	if (button) {
		button.addEventListener('click', (e) => {
			e.preventDefault();
			if (!isGenerating) {
				speakSelectedText();
			}
		});
	}

	// Update button state when selection changes
	document.addEventListener('selectionchange', () => {
		updateButtonState();
	});

	// Update button state when clicking anywhere
	document.addEventListener('mouseup', () => {
		setTimeout(updateButtonState, 10);
	});

	// Update button state on page load
	updateButtonState();

	// Cleanup on page unload
	window.addEventListener('beforeunload', () => {
		if (currentAudio) {
			currentAudio.pause();
			currentAudio = null;
		}
	});
})();

