<?php
/**
 * Plugin Name:       Kokoreader
 * Plugin URI:        https://kokoro.lancesmith.cc/wpplugins/kokoreader
 * Description:       Floating "Speak" button that reads selected text aloud using Kokoro AI
 * Version:           1.0.0
 * Author:            LanceSmith.cc
 * License:           GPLv2 or later
 * Text Domain:       kokoreader
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Kokoreader' ) ) {
	final class Kokoreader {
		const VERSION    = '1.0.0';
		const OPTION_KEY = 'kokoreader_settings';

		public static function init() {
			static $instance = null;
			if ( null === $instance ) {
				$instance = new self();
			}
		}

		private function __construct() {
			add_action( 'admin_menu', [ $this, 'register_settings_page' ] );
			add_action( 'admin_init', [ $this, 'register_settings' ] );
			add_action( 'wp_footer', [ $this, 'render_floating_button' ] );
			add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_assets' ] );
			add_action( 'rest_api_init', [ $this, 'register_rest_routes' ] );
		}

		public static function get_settings() {
			$defaults = [
				'enabled'   => '1',
				'voice'     => 'af_heart',
				'speed'     => '1.0',
				'position'  => 'bottom-right',
			];

			return wp_parse_args( get_option( self::OPTION_KEY, [] ), $defaults );
		}

		public function register_settings_page() {
			add_options_page(
				__( 'Kokoreader Settings', 'kokoreader' ),
				__( 'Kokoreader', 'kokoreader' ),
				'manage_options',
				'kokoreader-settings',
				[ $this, 'render_settings_page' ]
			);
		}

		public function register_settings() {
			register_setting( self::OPTION_KEY, self::OPTION_KEY, [ $this, 'sanitize_settings' ] );

			add_settings_section(
				'kokoreader_section',
				__( 'Reader Configuration', 'kokoreader' ),
				function () {
					echo '<p>' . esc_html__( 'Configure the floating "Speak" button that reads selected text.', 'kokoreader' ) . '</p>';
				},
				'kokoreader_settings'
			);

			add_settings_field(
				'kokoreader_enabled',
				__( 'Enable Reader', 'kokoreader' ),
				[ $this, 'render_checkbox_field' ],
				'kokoreader_settings',
				'kokoreader_section',
				[
					'label_for'  => 'kokoreader_enabled',
					'option_key' => 'enabled',
					'description' => __( 'Show the floating "Speak" button site-wide', 'kokoreader' ),
				]
			);

			add_settings_field(
				'kokoreader_voice',
				__( 'Voice', 'kokoreader' ),
				[ $this, 'render_select_field' ],
				'kokoreader_settings',
				'kokoreader_section',
				[
					'label_for'  => 'kokoreader_voice',
					'option_key' => 'voice',
					'options'    => $this->get_voice_options(),
				]
			);

			add_settings_field(
				'kokoreader_speed',
				__( 'Speed', 'kokoreader' ),
				[ $this, 'render_number_field' ],
				'kokoreader_settings',
				'kokoreader_section',
				[
					'label_for'  => 'kokoreader_speed',
					'option_key' => 'speed',
					'min'        => '0.5',
					'max'        => '2.0',
					'step'       => '0.1',
				]
			);

			add_settings_field(
				'kokoreader_position',
				__( 'Button Position', 'kokoreader' ),
				[ $this, 'render_select_field' ],
				'kokoreader_settings',
				'kokoreader_section',
				[
					'label_for'  => 'kokoreader_position',
					'option_key' => 'position',
					'options'    => [
						'bottom-right' => __( 'Bottom Right', 'kokoreader' ),
						'bottom-left'  => __( 'Bottom Left', 'kokoreader' ),
						'top-right'    => __( 'Top Right', 'kokoreader' ),
						'top-left'     => __( 'Top Left', 'kokoreader' ),
					],
				]
			);
		}

		protected function get_voice_options() {
			$voices = [
				// English (US) - Female
				'af_heart'     => __( 'Heart - US Female (Best Quality)', 'kokoreader' ),
				'af_bella'     => __( 'Bella - US Female', 'kokoreader' ),
				'af_nicole'    => __( 'Nicole - US Female', 'kokoreader' ),
				'af_sarah'     => __( 'Sarah - US Female', 'kokoreader' ),
				'af_aoede'     => __( 'Aoede - US Female', 'kokoreader' ),
				'af_kore'      => __( 'Kore - US Female', 'kokoreader' ),
				'af_alloy'     => __( 'Alloy - US Female', 'kokoreader' ),
				'af_nova'      => __( 'Nova - US Female', 'kokoreader' ),
				'af_sky'       => __( 'Sky - US Female', 'kokoreader' ),
				'af_jessica'   => __( 'Jessica - US Female', 'kokoreader' ),
				'af_river'     => __( 'River - US Female', 'kokoreader' ),
				
				// English (US) - Male
				'am_fenrir'    => __( 'Fenrir - US Male', 'kokoreader' ),
				'am_michael'   => __( 'Michael - US Male', 'kokoreader' ),
				'am_puck'      => __( 'Puck - US Male', 'kokoreader' ),
				'am_echo'      => __( 'Echo - US Male', 'kokoreader' ),
				'am_eric'      => __( 'Eric - US Male', 'kokoreader' ),
				'am_liam'      => __( 'Liam - US Male', 'kokoreader' ),
				'am_onyx'      => __( 'Onyx - US Male', 'kokoreader' ),
				'am_santa'     => __( 'Santa - US Male', 'kokoreader' ),
				'am_adam'      => __( 'Adam - US Male', 'kokoreader' ),
				
				// English (GB) - Female
				'bf_emma'      => __( 'Emma - British Female', 'kokoreader' ),
				'bf_isabella'  => __( 'Isabella - British Female', 'kokoreader' ),
				'bf_alice'     => __( 'Alice - British Female', 'kokoreader' ),
				'bf_lily'      => __( 'Lily - British Female', 'kokoreader' ),
				
				// English (GB) - Male
				'bm_george'    => __( 'George - British Male', 'kokoreader' ),
				'bm_fable'     => __( 'Fable - British Male', 'kokoreader' ),
				'bm_lewis'     => __( 'Lewis - British Male', 'kokoreader' ),
				'bm_daniel'    => __( 'Daniel - British Male', 'kokoreader' ),
				
				// Spanish
				'ef_dora'      => __( 'Dora - Spanish Female', 'kokoreader' ),
				'em_alex'      => __( 'Alex - Spanish Male', 'kokoreader' ),
				'em_santa'     => __( 'Santa - Spanish Male', 'kokoreader' ),
				
				// Japanese
				'jf_alpha'     => __( 'Alpha - Japanese Female', 'kokoreader' ),
				'jf_gongitsune'=> __( 'Gongitsune - Japanese Female', 'kokoreader' ),
				'jf_tebukuro'  => __( 'Tebukuro - Japanese Female', 'kokoreader' ),
				'jf_nezumi'    => __( 'Nezumi - Japanese Female', 'kokoreader' ),
				'jm_kumo'      => __( 'Kumo - Japanese Male', 'kokoreader' ),
				
				// Chinese (Mandarin)
				'zf_xiaobei'   => __( 'Xiaobei - Chinese Female', 'kokoreader' ),
				'zf_xiaoni'    => __( 'Xiaoni - Chinese Female', 'kokoreader' ),
				'zf_xiaoxiao'  => __( 'Xiaoxiao - Chinese Female', 'kokoreader' ),
				'zf_xiaoyi'    => __( 'Xiaoyi - Chinese Female', 'kokoreader' ),
				'zm_yunjian'   => __( 'Yunjian - Chinese Male', 'kokoreader' ),
				'zm_yunxi'     => __( 'Yunxi - Chinese Male', 'kokoreader' ),
				'zm_yunxia'    => __( 'Yunxia - Chinese Male', 'kokoreader' ),
				'zm_yunyang'   => __( 'Yunyang - Chinese Male', 'kokoreader' ),
				
				// Hindi
				'hf_alpha'     => __( 'Alpha - Hindi Female', 'kokoreader' ),
				'hf_beta'      => __( 'Beta - Hindi Female', 'kokoreader' ),
				'hm_omega'     => __( 'Omega - Hindi Male', 'kokoreader' ),
				'hm_psi'       => __( 'Psi - Hindi Male', 'kokoreader' ),
				
				// Italian
				'if_sara'      => __( 'Sara - Italian Female', 'kokoreader' ),
				'im_nicola'    => __( 'Nicola - Italian Male', 'kokoreader' ),
				
				// Portuguese
				'pf_dora'      => __( 'Dora - Portuguese Female', 'kokoreader' ),
				'pm_alex'      => __( 'Alex - Portuguese Male', 'kokoreader' ),
				'pm_santa'     => __( 'Santa - Portuguese Male', 'kokoreader' ),
			];

			return apply_filters( 'kokoreader_voice_options', $voices );
		}

		public function render_checkbox_field( $args ) {
			$settings = self::get_settings();
			$value    = $settings[ $args['option_key'] ] ?? '';
			$checked  = ! empty( $value );
			?>
			<label>
				<input
					type="checkbox"
					id="<?php echo esc_attr( $args['label_for'] ); ?>"
					name="<?php echo esc_attr( self::OPTION_KEY . '[' . $args['option_key'] . ']' ); ?>"
					value="1"
					<?php checked( $checked ); ?>
				/>
				<?php if ( ! empty( $args['description'] ) ) : ?>
					<span class="description"><?php echo esc_html( $args['description'] ); ?></span>
				<?php endif; ?>
			</label>
			<?php
		}

		public function render_select_field( $args ) {
			$settings = self::get_settings();
			$value    = $settings[ $args['option_key'] ] ?? '';
			?>
			<select
				id="<?php echo esc_attr( $args['label_for'] ); ?>"
				name="<?php echo esc_attr( self::OPTION_KEY . '[' . $args['option_key'] . ']' ); ?>"
			>
				<?php foreach ( $args['options'] as $key => $label ) : ?>
					<option value="<?php echo esc_attr( $key ); ?>" <?php selected( $value, $key ); ?>>
						<?php echo esc_html( $label ); ?>
					</option>
				<?php endforeach; ?>
			</select>
			<?php
		}

		public function render_number_field( $args ) {
			$settings = self::get_settings();
			$value    = $settings[ $args['option_key'] ] ?? '';
			?>
			<input
				type="number"
				id="<?php echo esc_attr( $args['label_for'] ); ?>"
				name="<?php echo esc_attr( self::OPTION_KEY . '[' . $args['option_key'] . ']' ); ?>"
				value="<?php echo esc_attr( $value ); ?>"
				min="<?php echo esc_attr( $args['min'] ?? '' ); ?>"
				max="<?php echo esc_attr( $args['max'] ?? '' ); ?>"
				step="<?php echo esc_attr( $args['step'] ?? 'any' ); ?>"
				style="width: 80px;"
			/>
			<?php
		}

		public function sanitize_settings( $input ) {
			$sanitized = [];

			$sanitized['enabled']  = ! empty( $input['enabled'] ) ? '1' : '';
			$sanitized['voice']    = sanitize_text_field( $input['voice'] ?? 'af_heart' );
			$sanitized['speed']    = floatval( $input['speed'] ?? 1.0 );
			$sanitized['position'] = sanitize_text_field( $input['position'] ?? 'bottom-right' );

			// Validate speed range
			$sanitized['speed'] = max( 0.5, min( 2.0, $sanitized['speed'] ) );

			return $sanitized;
		}

		public function render_settings_page() {
			if ( ! current_user_can( 'manage_options' ) ) {
				return;
			}
			?>
			<div class="wrap">
				<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
				<form method="post" action="options.php">
					<?php
					settings_fields( self::OPTION_KEY );
					do_settings_sections( 'kokoreader_settings' );
					submit_button();
					?>
				</form>

				<div style="margin-top: 30px; padding: 15px; background: #f0f0f1; border-left: 4px solid #2271b1;">
					<h2><?php esc_html_e( 'How to Use', 'kokoreader' ); ?></h2>
					<ol>
						<li><?php esc_html_e( 'Enable the reader above', 'kokoreader' ); ?></li>
						<li><?php esc_html_e( 'Choose your preferred voice and speed', 'kokoreader' ); ?></li>
						<li><?php esc_html_e( 'Save settings', 'kokoreader' ); ?></li>
						<li><?php esc_html_e( 'Visit any page on your site', 'kokoreader' ); ?></li>
						<li><?php esc_html_e( 'Select text on the page', 'kokoreader' ); ?></li>
						<li><?php esc_html_e( 'Click the floating "Speak" button', 'kokoreader' ); ?></li>
						<li><?php esc_html_e( 'The selected text will be read aloud!', 'kokoreader' ); ?></li>
					</ol>
				</div>
			</div>
			<?php
		}

		public function enqueue_assets() {
			$settings = self::get_settings();

			// Only load if enabled
			if ( empty( $settings['enabled'] ) ) {
				return;
			}

			wp_enqueue_style(
				'kokoreader-frontend',
				plugin_dir_url( __FILE__ ) . 'assets/css/kokoreader.css',
				[],
				self::VERSION
			);

			wp_enqueue_script(
				'kokoreader-frontend',
				plugin_dir_url( __FILE__ ) . 'assets/js/kokoreader.js',
				[],
				self::VERSION,
				true
			);

			wp_localize_script(
				'kokoreader-frontend',
				'kokoreaderSettings',
				[
					'restUrl'  => esc_url_raw( rest_url( 'kokoreader/v1/speak' ) ),
					'voice'    => $settings['voice'],
					'speed'    => floatval( $settings['speed'] ),
					'position' => $settings['position'],
				]
			);
		}

		public function render_floating_button() {
			$settings = self::get_settings();

			// Only render if enabled
			if ( empty( $settings['enabled'] ) ) {
				return;
			}

			?>
			<div id="kokoreader-floating-button" class="kokoreader-position-<?php echo esc_attr( $settings['position'] ); ?>">
				<button type="button" class="kokoreader-button" aria-label="<?php esc_attr_e( 'Read selected text', 'kokoreader' ); ?>">
					<svg class="kokoreader-icon-speak" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
						<polygon points="11 5 6 9 2 9 2 15 6 15 11 19 11 5"></polygon>
						<path d="M15.54 8.46a5 5 0 0 1 0 7.07"></path>
					</svg>
					<span class="kokoreader-icon-loading">↻</span>
					<span><?php esc_html_e( 'Speak', 'kokoreader' ); ?></span>
				</button>
				<div class="kokoreader-status"></div>
			</div>
			<?php
		}

		public function register_rest_routes() {
			register_rest_route(
				'kokoreader/v1',
				'/speak',
				[
					'methods'             => 'POST',
					'callback'            => [ $this, 'rest_speak' ],
					'permission_callback' => '__return_true', // Public access
					'args'                => [
						'text' => [
							'required' => true,
							'type'     => 'string',
							'validate_callback' => function( $value ) {
								return ! empty( trim( $value ) );
							},
						],
					],
				]
			);
		}

		public function rest_speak( $request ) {
			$text     = trim( $request->get_param( 'text' ) );
			$settings = self::get_settings();

			if ( empty( $text ) ) {
				return new WP_Error( 'no_text', __( 'No text provided', 'kokoreader' ), [ 'status' => 400 ] );
			}

			// Limit text length
			if ( strlen( $text ) > 5000 ) {
				$text = substr( $text, 0, 5000 );
			}

			// Call Kokoro API
			// Longer timeout for large text (up to 5 minutes)
			$response = wp_remote_post(
				'https://kokoro.lancesmith.cc/api/v1/audio/speech',
				[
					'timeout' => 300, // 5 minutes for long text generation
					'headers' => [
						'Content-Type' => 'application/json',
					],
					'body'    => wp_json_encode(
						[
							'model'           => 'model_q8f16',
							'voice'           => $settings['voice'],
							'input'           => $text,
							'response_format' => 'mp3',
							'speed'           => floatval( $settings['speed'] ),
						]
					),
				]
			);

			if ( is_wp_error( $response ) ) {
				return new WP_Error( 'api_error', $response->get_error_message(), [ 'status' => 500 ] );
			}

			$status_code = wp_remote_retrieve_response_code( $response );
			if ( $status_code !== 200 ) {
				return new WP_Error( 'api_error', __( 'Failed to generate speech', 'kokoreader' ), [ 'status' => $status_code ] );
			}

			$audio_data = wp_remote_retrieve_body( $response );

			// Return audio as base64
			return rest_ensure_response(
				[
					'audio'     => base64_encode( $audio_data ),
					'mime_type' => 'audio/mpeg',
				]
			);
		}
	}

	add_action( 'plugins_loaded', [ 'Kokoreader', 'init' ] );
}

